// ============================================================================
// ============================================================================
// ============================================================================
// ==                                                                        ==
// == Name    : TheEmuLib.Pulse_RGBA.B.16.fsh                                ==
// == Type    : Fragment shader                                              ==
// == Version : 1.0.0 (2017/02/13)                                           ==
// == Creator : TheEmu © TheEmu 2017, Some Rights Reserved                   ==
// == Licence : Creative Commons Attribution-ShareAlike 4.0                  ==
// ==           http://creativecommons.org/licences/by-sa/4.0                ==
// ==                                                                        ==
// == Purpose: Apply pulses of colour or opacity to a source image.          ==
// ==                                                                        ==
// == Description: The Red, Green, Blue and Alpha channels of a source image ==
// == are subjected to "pulses" according to a schedule that is defined by a ==
// == set of uniform variables declared that may be declared in a .scn file. ==
// ==                                                                        ==
// == A pulse comprises four phases                                          ==
// ==                                                                        ==
// ==   A leading edge     - the pulse level is rising                       ==
// ==   A stable phase     - the pulse level is sustained                    ==
// ==   A trailing edge    - the pulse level is falling                      ==
// ==   A post pulse phase - waiting for the next pulse                      ==
// ==                                                                        ==
// == The overall pulse schedule also has four phases                        ==
// ==                                                                        ==
// ==   An initial quiescent phase with no pulses                            ==
// ==   An optional initial sequence of non-cyclic pulses                    ==
// ==   An optional cyclic period when pulses are repeated                   ==
// ==   An optional post-cyclic period with non-cyclic pulses                ==
// ==                                                                        ==
// == The pulse pattern may optionaly be inverted.                           ==
// ==                                                                        ==
// == This version of the shader supports different "high" and "low"  levels ==
// == for each of the pulses and up to 16 pulse definitions.  Other versions ==
// == of the shader impose different limits on the number of pulses and / or ==
// == restrictions on the pulse levels.  For the best performance you should ==
// == use the smallest version that suits your particular needs.             ==
// ==                                                                        ==
// == This file is a member of The Emu's shader library.                     ==
// ==                                                                        ==
// == ====================================================================== ==
// ==                                                                        ==
// == Update history:                                                        ==
// ==                                                                        ==
// ==   2017/02/15 - v1.0.0 - Initial version                                ==
// ==                                                                        ==
// ============================================================================
// ============================================================================
// ============================================================================

// ============================================================================
// == Standard shader inputs ==================================================
// ============================================================================

uniform float u_Elapsed;    // The elapsed time in seconds
uniform vec2  u_WindowSize; // Window dimensions in pixels

// The image that is to be manipulated.

uniform sampler2D iChannel0;

// ============================================================================
// == Imports from TheEmuLib ==================================================
// ============================================================================
//
// The GLSL shader language currently provides no mechanism for importing  any
// elements that are defined in other modules, not even C's crude source level
// #include mechanism. In the absence of anything better TheEmuLib handles any
// imports by manualy copying relevent utility code snippets from the  sources
// in the Shader Lib.Inc directory. This is very crude but I have attempted to
// be systematic in the way in which this is presented in the library sources.
//
// = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = =

// Macros from TheEmuLib.Emu_Common_Utilities.lib.src

#define EMU_DEFAULT(type,x,default_value) ( (x==type(0.0)) ? (default_value) : (x) )

// = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = =

// Functions from TheEmuLib.Emu_Coordinate_Normalisation.lib.src

#define EMU_NORMALISE_TO_WINDOW_1(xy,wsize) ( (xy)/(wsize) )

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

vec2 Emu_Normalise_to_Window ( vec2 xy )
 { return EMU_NORMALISE_TO_WINDOW_1 ( xy, u_WindowSize.xy );
 }

// ============================================================================
// == Shader specific inputs ==================================================
// ============================================================================

// The following parameter controls the overall mode of the shader. Currently
// there is only one aspect that it controls, namely whether or not the sense
// of the pulses is to be inverted or not.  This  is  controlled by the least
// significant bit of the parameter.

uniform int Emu_Pulse_RGBA_mode;

bool invert = ( Emu_Pulse_RGBA_mode & 1 ) != 0;

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
//
// Each pulse is defined by three vec4 parameters,  for  pulse number nn these
// parameters are Tnn, Ann and Bnn where nn = 00, 01, 02 etc.
//
//    Tnn - pulse times
//    Ann - RGBA components of pulse level A
//    Ann - RGBA components of pulse level B
//
// Each pulse is regarded as having three phases, a leading edge,  a sustained
// middle and a trailing edege.  During the leading edge phase the pulse value
// changes from its that given by the corresponding A parameter to that  given
// by the corresponding B parameter,  during  the  sustained  middle phase the
// pulse intensity is constant and during the trailing edge phase the  pulse's
// intensity goes from its B value to the A value of the following pulse.
//
// It is expected that in normal use te A and B parameters will  be  the  same
// for all pulses, but other usage patterns are allowed, e.g. staircases.
//
// By default the pulses are "positive" going with the A parameters specifying
// the pulses "off" or "low" levels while the B parameters specify the "on" or
// "high" levels.  Although the levels are described as "low" and "high" it is
// quite valid to use "high" levels that are lower than the "low" levels. 
//
// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
//
// The four components of Tnn are
//
//    Time at which pulse nn's leading edge starts 
//    Time at which pulse nn's leading edge ends
//    Time at which pulse nn's trailing edge start
//    Time at which pulse nn's trailing edge ends
//
// However for square pulses only the overall pulse start and end times need
// be specified and for pulses that have no sustained pulse, i.e. pulses for
// which the trailing edge starts as soon as the leading edge finishes, only
// the pulse start, pulse mid-point and pulse end times need be provided and
// the shader will fill in the missing times using the following scheme
//
//     t0,t1       => t0,t0, t1,t1 - square ended pulse
//     t0,t1,t2    => t0,t1, t1,t2 - no sustained phase
//     t0,t1,t2,t3 => t0,t1, t2,t3 - full 3 phase pulse
//
// where missing trailing values are detected by being zero or negative. With
// the exception of trailing zero special cases the times in each Tnn  should
// be in ascending order and the Tnn's themselves be in ascending time order.

uniform vec4 Emu_Pulse_RGBA_T01;
uniform vec4 Emu_Pulse_RGBA_T02;
uniform vec4 Emu_Pulse_RGBA_T03;
uniform vec4 Emu_Pulse_RGBA_T04;
uniform vec4 Emu_Pulse_RGBA_T05;
uniform vec4 Emu_Pulse_RGBA_T06;
uniform vec4 Emu_Pulse_RGBA_T07;
uniform vec4 Emu_Pulse_RGBA_T08;
uniform vec4 Emu_Pulse_RGBA_T09;
uniform vec4 Emu_Pulse_RGBA_T10;
uniform vec4 Emu_Pulse_RGBA_T11;
uniform vec4 Emu_Pulse_RGBA_T12;

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

// The Ann and Bnn parameters define the high and low levels reached for the
// red, green, blue and alpha (opacity) components of each pulse.  The full,
// B, versions of the shader allow the levels to be defined  separately  for
// each pulse as well as global pair of levels. The A versions of the shader
// only support specifying the global levels.  The  global levels simply act
// as the defaults for the individual pulse levels.

uniform vec4 Emu_Pulse_RGBA_A00; // "Global" pulse level 
uniform vec4 Emu_Pulse_RGBA_B00; // "Global" pulse level 

uniform vec4 Emu_Pulse_RGBA_A01;
uniform vec4 Emu_Pulse_RGBA_A02;
uniform vec4 Emu_Pulse_RGBA_A03;
uniform vec4 Emu_Pulse_RGBA_A04;
uniform vec4 Emu_Pulse_RGBA_A05;
uniform vec4 Emu_Pulse_RGBA_A06;
uniform vec4 Emu_Pulse_RGBA_A07;
uniform vec4 Emu_Pulse_RGBA_A08;
uniform vec4 Emu_Pulse_RGBA_A09;
uniform vec4 Emu_Pulse_RGBA_A10;
uniform vec4 Emu_Pulse_RGBA_A11;
uniform vec4 Emu_Pulse_RGBA_A12;

uniform vec4 Emu_Pulse_RGBA_B01;
uniform vec4 Emu_Pulse_RGBA_B02;
uniform vec4 Emu_Pulse_RGBA_B03;
uniform vec4 Emu_Pulse_RGBA_B04;
uniform vec4 Emu_Pulse_RGBA_B05;
uniform vec4 Emu_Pulse_RGBA_B06;
uniform vec4 Emu_Pulse_RGBA_B07;
uniform vec4 Emu_Pulse_RGBA_B08;
uniform vec4 Emu_Pulse_RGBA_B09;
uniform vec4 Emu_Pulse_RGBA_B10;
uniform vec4 Emu_Pulse_RGBA_B11;
uniform vec4 Emu_Pulse_RGBA_B12;

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

// By default the pulse pattern is not cyclic.  However the following control
// parameter can be used to change this to a pattern where a number of pulses
// is an initial phase of pulses that are only generated once followed  by  a
// phase in which the pulses are generated cyclicly.  It  is even possible to
// specify a final phase where the pulses are once more non-cyclic.
//
// The components of Emu_Pulse_RGBA_cycle_times are
//
//     T0 - The time for which pulse generation is suspended
//     T1 - The time at which cyclic pulse generation begins
//     T2 - The duration of a single cycle of pulse generation
//     T3 - The duration of cyclic pulse generation
//
// T1 is relative to T0, T2 and T3 are durations but they can equally well be
// regarded as times relative to T1.

uniform vec4 Emu_Pulse_RGBA_cycle;

vec4 cyc = EMU_DEFAULT ( vec4, Emu_Pulse_RGBA_cycle, vec4(0.0,0.0,1.0e50,1.0e50) );
vec4 cycle = vec4 ( cyc.xyz, EMU_DEFAULT(float,cyc.w,1.0e50) );

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

// The time used in the shader may be sped up or slowed down by an  arbitrary
// factor. This is primarily intended to be used during scene development and
// testing as a way of getting through test quickly,  but there is nothing to
// stop it being used in a released scene, other than perhaps "good taste" in
// programming.

uniform float Emu_Pulse_RGBA_speed;

float time = u_Elapsed * EMU_DEFAULT ( float, Emu_Pulse_RGBA_speed, 1.0 );

// ============================================================================

// Prepare the internal forms used for the Tnn, Ann and Bnn parameters.

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

// Macros to recognise the special forms for Tnn that can be used for square
// ended pulses and for those with no sustained phase.

#define NOT_DEFINED(t)   ( all ( lessThanEqual( t,    vec4(0.0) ) ) )
#define SQUARE_ENDED(t)  ( all ( lessThanEqual( t.zw, vec2(0.0) ) ) )
#define NOT_SUSTAINED(t) ( t.w <= 0.0 )

// A macro to convert any special Tnn forms to the normal canonical form.

#define NORMALISE_Tnn(t) ( NOT_DEFINED(t)   ? vec4(1.0e44) \
                         : SQUARE_ENDED(t)  ? t.xxyy       \
                         : NOT_SUSTAINED(t) ? t.xyyz       \
                         :                    t.xyzw       \
                         )                                 \

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

// Normalise and assemble the Tnn, Ann and Bnn parameters into  arrays  that 
// are used in the body of the shader.

#define MAX_ix 12 // The maximum number of user definable pulses.

// The pulse schedule. The first and last entries simplify the logic used in
// the shader by eliminating special cases at the schedule start and end.

#define Txx vec4(1.0e44)
#define Tzz vec4(1.0e44)

vec4[MAX_ix+2] Tnn = vec4[MAX_ix+2] ( Txx,
                                      NORMALISE_Tnn ( Emu_Pulse_RGBA_T01 ),
                                      NORMALISE_Tnn ( Emu_Pulse_RGBA_T02 ),
                                      NORMALISE_Tnn ( Emu_Pulse_RGBA_T03 ),
                                      NORMALISE_Tnn ( Emu_Pulse_RGBA_T04 ),
                                      NORMALISE_Tnn ( Emu_Pulse_RGBA_T05 ),
                                      NORMALISE_Tnn ( Emu_Pulse_RGBA_T06 ),
                                      NORMALISE_Tnn ( Emu_Pulse_RGBA_T07 ),
                                      NORMALISE_Tnn ( Emu_Pulse_RGBA_T08 ),
                                      NORMALISE_Tnn ( Emu_Pulse_RGBA_T09 ),
                                      NORMALISE_Tnn ( Emu_Pulse_RGBA_T10 ),
                                      NORMALISE_Tnn ( Emu_Pulse_RGBA_T11 ),
                                      NORMALISE_Tnn ( Emu_Pulse_RGBA_T12 ),
                                      Tzz
                                    );

// The pulses' low and high levels. The 00 pulse number indicates that the
// levels apply to all pulses unless individualy overridden.

vec4 A00 = EMU_DEFAULT ( vec4, Emu_Pulse_RGBA_A00, vec4(0.0) );
vec4 B00 = EMU_DEFAULT ( vec4, Emu_Pulse_RGBA_B00, vec4(1.0) );

#define Axx vec4(0.0)
#define Azz vec4(0.0)

vec4[MAX_ix+2] Ann = vec4[MAX_ix+2] ( Axx,
                                      EMU_DEFAULT ( vec4, Emu_Pulse_RGBA_A01, A00 ),
                                      EMU_DEFAULT ( vec4, Emu_Pulse_RGBA_A02, A00 ),
                                      EMU_DEFAULT ( vec4, Emu_Pulse_RGBA_A03, A00 ),
                                      EMU_DEFAULT ( vec4, Emu_Pulse_RGBA_A04, A00 ),
                                      EMU_DEFAULT ( vec4, Emu_Pulse_RGBA_A05, A00 ),
                                      EMU_DEFAULT ( vec4, Emu_Pulse_RGBA_A06, A00 ),
                                      EMU_DEFAULT ( vec4, Emu_Pulse_RGBA_A07, A00 ),
                                      EMU_DEFAULT ( vec4, Emu_Pulse_RGBA_A08, A00 ),
                                      EMU_DEFAULT ( vec4, Emu_Pulse_RGBA_A09, A00 ),
                                      EMU_DEFAULT ( vec4, Emu_Pulse_RGBA_A10, A00 ),
                                      EMU_DEFAULT ( vec4, Emu_Pulse_RGBA_A11, A00 ),
                                      EMU_DEFAULT ( vec4, Emu_Pulse_RGBA_A12, A00 ),
                                      Azz
                                    );

#define Bxx vec4(1.0)
#define Bzz vec4(1.0)

vec4[MAX_ix+2] Bnn = vec4[MAX_ix+2] ( Bxx,
                                      EMU_DEFAULT ( vec4, Emu_Pulse_RGBA_B01, B00 ),
                                      EMU_DEFAULT ( vec4, Emu_Pulse_RGBA_B02, B00 ),
                                      EMU_DEFAULT ( vec4, Emu_Pulse_RGBA_B03, B00 ),
                                      EMU_DEFAULT ( vec4, Emu_Pulse_RGBA_B04, B00 ),
                                      EMU_DEFAULT ( vec4, Emu_Pulse_RGBA_B05, B00 ),
                                      EMU_DEFAULT ( vec4, Emu_Pulse_RGBA_B06, B00 ),
                                      EMU_DEFAULT ( vec4, Emu_Pulse_RGBA_B07, B00 ),
                                      EMU_DEFAULT ( vec4, Emu_Pulse_RGBA_B08, B00 ),
                                      EMU_DEFAULT ( vec4, Emu_Pulse_RGBA_B09, B00 ),
                                      EMU_DEFAULT ( vec4, Emu_Pulse_RGBA_B10, B00 ),
                                      EMU_DEFAULT ( vec4, Emu_Pulse_RGBA_B11, B00 ),
                                      EMU_DEFAULT ( vec4, Emu_Pulse_RGBA_B12, B00 ),
                                      Bzz
                                    );

// ============================================================================
// == The shader's main routine ===============================================
// ============================================================================

void main ( void )
 {
   // Determine the basic time parameters
   //
   //    time_A - elapsed time since time released
   //    time_B - time since start of cyclic phase
   //    time_C - time since start of current cycle 
   //    time_D - time since end of cyclic phase

   float time_A = max ( time - cycle[0], 0.0 );
   float time_B = max (    time_A - cycle[1], 0.0 );
   float time_C = mod (    time_B,  cycle[2]      );
   float time_D = max (    time_B - cycle[3], 0.0 );
 
   // Modify time_A to take any cyclic phase into account.

   if ( time_D > 0.0 ) time_C = -1.0;
   if ( time_C > 0.0 ) time_A = time_C + cycle[1];

   // Find the pulse definition for the current time.  The 
   // method used tries to test all of the possible pulses
   // in parallel which should be faster than any search.

   // Check each possible pulse. If the schedule has been
   // properly defined there should be at most one match.

   #define in_Tnn(ii) ( (time_A>=Tnn[ii][0]) && (time_A<Tnn[ii+1][0]) )

   vec4[3] tnn_match = vec4[3] ( vec4 ( in_Tnn(1), in_Tnn(2),  in_Tnn(3),  in_Tnn(4)  ),
                                 vec4 ( in_Tnn(5), in_Tnn(6),  in_Tnn(7),  in_Tnn(8)  ),
                                 vec4 ( in_Tnn(9), in_Tnn(10), in_Tnn(11), in_Tnn(12) )
                               );

   // If there is only one match then the following statement
   // will identify it and if there are no matches it  yeilds
   // zero.  If there are more than one matches then there is
   // an error in the schedule definition and the result will
   // not be meaningfull, but no error checking is done. 

   int ix = int ( dot ( tnn_match[0], vec4 ( 1.0,  2.0,  3.0,  4.0) )
                + dot ( tnn_match[1], vec4 ( 5.0,  6.0,  7.0,  8.0) )
                + dot ( tnn_match[2], vec4 ( 9.0, 10.0, 11.0, 12.0) )
                );

   // The index ix identifies the Tnn either for the currently
   // active pulse or the previous one if no pulse is active.

   float pulse = smoothstep ( Tnn[ix][0], Tnn[ix][1], time_A )
       * ( 1.0 - smoothstep ( Tnn[ix][2], Tnn[ix][3], time_A ) );

   // Optionaly invert the pulse.

   if ( invert ) pulse = 1.0 - pulse;

   // In the inter-pulse gaps ramp the pulse's low level between
   // those applicable to the previous pulse and its successor.

   float f = ( time_A - Tnn[ix][3] ) / ( Tnn[ix+1][3] - Tnn[ix][3]  );
   vec4  A = mix ( Ann[ix], Ann[ix+1], clamp(0.0,1.0,f) );

   // Apply the pulse to the pixel colour from the source image.

   vec2 uv = Emu_Normalise_to_Window ( gl_FragCoord.xy );

   vec4 colour = texture2D(iChannel0,uv) * mix(A,Bnn[ix],pulse);

   // If debugging is enabled the bottom part of the screen is used
   // to show information I needed to debug this shader.  Debugging
   // is enabled by uncommenting the #define DEBUG statement.  This
   // debugging code has been left in as an example of how to debug
   // shaders rather than as something necessary for this one. This
   // debug code is deliberately simple and not very efficient,  it
   // is likely to badly affect the shader's performance. It should
   // only be enabled if you need it.

// #define DEBUG
   #ifdef  DEBUG

      if ( uv.y > 0.97 ) {

         vec4 dbg_red = vec4(1.0,0.0,0.0,1.0);

         // Three bands of colour are used.  The top one shows which
         // Tnn's were matched by each test in the search.  Only one
         // match should be found if the pulse schedule is valid and
         // no more than a single block of red shown in this part of
         // the debugging output.

         switch ( int ( uv.x * 12.0 ) )
          {  case  0: colour = dbg_red * float(tnn_match[0][0]); break;
             case  1: colour = dbg_red * float(tnn_match[0][1]); break;
             case  2: colour = dbg_red * float(tnn_match[0][2]); break;
             case  3: colour = dbg_red * float(tnn_match[0][3]); break;

             case  4: colour = dbg_red * float(tnn_match[1][0]); break;
             case  5: colour = dbg_red * float(tnn_match[1][1]); break;
             case  6: colour = dbg_red * float(tnn_match[1][2]); break;
             case  7: colour = dbg_red * float(tnn_match[1][3]); break;

             case  8: colour = dbg_red * float(tnn_match[2][0]); break;
             case  9: colour = dbg_red * float(tnn_match[2][1]); break;
             case 10: colour = dbg_red * float(tnn_match[2][2]); break;
             case 11: colour = dbg_red * float(tnn_match[2][3]); break;
          }

         if ( uv.y > 0.98 )
          {
            // Debugging displays the values of shader variables encoded
            // as easily interpretable colours.  The Red, Green and Blue
            // components of these colours are either 0 or 1 so we  have
            // eight colours which encode three bit binary integers.

           vec4[8] dbg_colour
              = vec4[8] ( vec4(0.0,0.0,0.0,1.0), // 000 - Black
                          vec4(0.0,0.0,1.0,1.0), // 001 - Blue
                          vec4(0.0,1.0,0.0,1.0), // 010 - Green
                          vec4(0.0,1.0,1.0,1.0), // 011 - Cyan
                          vec4(1.0,0.0,0.0,1.0), // 100 - Red
                          vec4(1.0,0.0,1.0,1.0), // 101 - Magenta
                          vec4(1.0,1.0,0.0,1.0), // 110 - Yellow
                          vec4(1.0,1.0,1.0,1.0)  // 111 - White
                        );

            // The second band shows the value of ix, i.e. which pulse
            // schedule entry was used and the third band displays the
            // time in units of 10 seconds. As was explained above the
            // colours are interpretable as binary numbers.

            int dbg_ix = (uv.y>0.99) ? int(floor(time/10.0)) : ix;
            dbg_ix = int ( mod ( float(dbg_ix), 8.0 ) );

            colour = dbg_colour[dbg_ix];

            // A moving dot also indicates the elapsed time.  This dot
            // will pass from left to right across the screen every 10
            // seconds and has the inverse colour of its background so
           // that it is always eaily visible. 

            vec2 qq = vec2 ( mod(time,10.0)/10.0, 0.99 );
   
            if ( length(qq-uv) < 0.0025 ) colour.rgb = 1.0 - colour.rgb;

         }
      }

   #endif

   // Update shader outputs.

   gl_FragColor = colour * gl_Color;

}